precision highp float;
varying vec2 vTextureCoord;
uniform sampler2D sTexture;

uniform float scaleFactor;
uniform float textureSize;
uniform float textureWidth;
uniform float textureHeight;

uniform sampler2D sTexture2;
uniform float texture2Size;
uniform float texture2Width;
uniform float texture2Height;
uniform int blendMode;
uniform float fade;


vec4 alphaBlend(vec4 src ,vec4 dst){
    vec4 result;
    
    float resultA = dst.a+src.a*(1.0-dst.a);
    
    if(resultA == 0.0){
        result = vec4(0.0,0.0,0.0,0.0);
    }else{
        result.r = (dst.r*dst.a+src.r*src.a*(1.0-dst.a))/resultA;
        result.g = (dst.g*dst.a+src.g*src.a*(1.0-dst.a))/resultA;
        result.b = (dst.b*dst.a+src.b*src.a*(1.0-dst.a))/resultA;
        result.a = resultA;
    }
    return result;
}

float screenValuesBlend(float srcValue,float destValue){
    srcValue *= 255.0;
    destValue *= 255.0; 
    float resultValue = (255.0 - (((255.0 - srcValue)*(255.0-destValue))/255.0));
    return resultValue/255.0;
}

vec4 screenBlend(vec4 originColor,vec4 maskColor){
	vec4 result;
	result.r = screenValuesBlend(originColor.r,maskColor.r);
	result.g = screenValuesBlend(originColor.g,maskColor.g);
	result.b = screenValuesBlend(originColor.b,maskColor.b);
	result.a = originColor.a;
	return result;
}

float overlayValuesBlend(float srcValue,float destValue){
    srcValue *= 255.0;
    destValue *= 255.0;
    
    float resultValue;
    
    if(srcValue>127.5)
	{
		float unitValue = (255.0-srcValue)/127.5;
		float minValue = srcValue - (255.0-srcValue);
		resultValue =  ((destValue * unitValue) + minValue);
	}
	else
	{
		float unitValue =srcValue/127.5;
		resultValue =  (destValue * unitValue);
	}
    return resultValue/255.0;
}

vec4 overlayBlend(vec4 originColor,vec4 maskColor){
	vec4 result;
	result.r = overlayValuesBlend(originColor.r,maskColor.r);
	result.g = overlayValuesBlend(originColor.g,maskColor.g);
	result.b = overlayValuesBlend(originColor.b,maskColor.b);
	result.a = originColor.a;
	return result;
}

float multiplyValuesBlend(float srcValue,float destValue){
    float result = srcValue*destValue;
    if(result<0.0){
        result = 0.0;
    }else if(result>1.0){
        result = 1.0;
    }
    return result;
}

vec4 multiplyBlend(vec4 originColor,vec4 maskColor){
	vec4 result;
	result.r = multiplyValuesBlend(originColor.r,maskColor.r);
	result.g = multiplyValuesBlend(originColor.g,maskColor.g);
	result.b = multiplyValuesBlend(originColor.b,maskColor.b);
	result.a = originColor.a;
	return result;
}


void main() {
	vec4 origColor = texture2D(sTexture, vTextureCoord);
	vec4 destColor;
	
	
	
	if(blendMode == 0){
		//original
		destColor = origColor;
	}else{
		if(textureSize>0.0 && texture2Size>0.0 && texture2Width>0.0 && texture2Height>0.0){
			float fXIndex = vTextureCoord.x*textureSize;
 			float fYIndex = vTextureCoord.y*textureSize;
 		
 			float xInBlendTexture = fXIndex/textureWidth*texture2Width/texture2Size;
 			float yInBlendTexture = fYIndex/textureHeight*texture2Height/texture2Size;
			vec4 maskColor = texture2D(sTexture2, vec2(xInBlendTexture,yInBlendTexture));
		
			if(blendMode == 1){
				//alpha
				destColor = alphaBlend(origColor,maskColor);
			}else if(blendMode == 2){
				//screen
				destColor = screenBlend(origColor,maskColor);
			}else if(blendMode == 3){
				//overlay
				destColor = overlayBlend(origColor,maskColor);
			}else if(blendMode == 4){
				//multiply
				destColor = multiplyBlend(origColor,maskColor);
			}else{
				destColor = maskColor;
			}
		
			float red = (fade*origColor.r+(100.0-fade)*destColor.r)/100.0;
			float green = (fade*origColor.g+(100.0-fade)*destColor.g)/100.0;
			float blue = (fade*origColor.b+(100.0-fade)*destColor.b)/100.0;
			destColor.r = red;
			destColor.g = green;
			destColor.b = blue;
		}else{
			destColor = origColor;
		}
	}
	gl_FragColor = destColor;  
}
	
	
